'use client';

import { useEffect, useState } from 'react';
import { useParams } from 'next/navigation';
import Link from 'next/link';
import Image from 'next/image';
import { supabase, Blog } from '@/lib/supabase';
import { Calendar, User, ArrowLeft, Share2, Linkedin, Twitter, Facebook } from 'lucide-react';
import { format } from 'date-fns';
import { Button } from '@/components/ui/button';
import { Skeleton } from '@/components/ui/skeleton';

export default function BlogDetailPage() {
  const params = useParams();
  const slug = decodeURIComponent(params.slug as string);
  const [blog, setBlog] = useState<Blog | null>(null);
  const [loading, setLoading] = useState(true);
  const [relatedBlogs, setRelatedBlogs] = useState<Blog[]>([]);

  useEffect(() => {
    async function fetchBlog() {
      const { data, error } = await supabase
        .from('blogs')
        .select('*')
        .eq('slug', slug)
        .eq('published', true)
        .single();

      if (!error && data) {
        setBlog(data as Blog);
        
        // Fetch related blogs
        const { data: related } = await supabase
          .from('blogs')
          .select('*')
          .eq('published', true)
          .neq('id', data.id)
          .order('published_at', { ascending: false })
          .limit(3);
        
        if (related) {
          setRelatedBlogs(related as Blog[]);
        }
      }
      setLoading(false);
    }
    fetchBlog();
  }, [slug]);

  const shareUrl = typeof window !== 'undefined' ? window.location.href : '';

  const handleShare = (platform: string) => {
    const title = blog?.title || '';
    let url = '';
    
    switch (platform) {
      case 'linkedin':
        url = `https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(shareUrl)}`;
        break;
      case 'twitter':
        url = `https://twitter.com/intent/tweet?text=${encodeURIComponent(title)}&url=${encodeURIComponent(shareUrl)}`;
        break;
      case 'facebook':
        url = `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(shareUrl)}`;
        break;
    }
    
    window.open(url, '_blank', 'width=600,height=400');
  };

  if (loading) {
    return (
      <main className="min-h-screen bg-background pt-24">
        <div className="container mx-auto px-4 py-12 max-w-4xl">
          <Skeleton className="h-8 w-32 mb-8" />
          <Skeleton className="h-12 w-full mb-4" />
          <Skeleton className="h-6 w-64 mb-8" />
          <Skeleton className="h-96 w-full mb-8 rounded-2xl" />
          <div className="space-y-4">
            <Skeleton className="h-4 w-full" />
            <Skeleton className="h-4 w-full" />
            <Skeleton className="h-4 w-3/4" />
          </div>
        </div>
      </main>
    );
  }

  if (!blog) {
    return (
      <main className="min-h-screen bg-background pt-24">
        <div className="container mx-auto px-4 py-16 text-center">
          <h1 className="text-3xl font-bold text-foreground mb-4">Blog Post Not Found</h1>
          <p className="text-muted-foreground mb-8">The article you're looking for doesn't exist or has been removed.</p>
          <Link href="/blog">
            <Button>
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Blog
            </Button>
          </Link>
        </div>
      </main>
    );
  }

  return (
    <main className="min-h-screen bg-background">
      {/* Hero Background */}
      <div className="bg-[#7D2C5A] h-32" />
      
      {/* Back Link */}
      <div className="container mx-auto px-4 py-6 max-w-4xl">
        <Link href="/blog" className="inline-flex items-center text-muted-foreground hover:text-primary transition-colors">
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Blog
        </Link>
      </div>

      {/* Article Header */}
      <article className="container mx-auto px-4 max-w-4xl">
        <header className="mb-8">
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold text-foreground mb-6 leading-tight">
            {blog.title}
          </h1>
          
          <div className="flex flex-wrap items-center gap-6 text-muted-foreground mb-6">
            <div className="flex items-center gap-3">
              {blog.author_avatar ? (
                <Image
                  src={blog.author_avatar}
                  alt={blog.author_name}
                  width={48}
                  height={48}
                  className="rounded-full"
                />
              ) : (
                <div className="w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center">
                  <User className="h-6 w-6 text-primary" />
                </div>
              )}
              <div>
                <p className="font-medium text-foreground">{blog.author_name}</p>
                {blog.published_at && (
                  <p className="text-sm flex items-center gap-1">
                    <Calendar className="h-3 w-3" />
                    {format(new Date(blog.published_at), 'MMMM d, yyyy')}
                  </p>
                )}
              </div>
            </div>
            
            {/* Share Buttons */}
            <div className="flex items-center gap-2 ml-auto">
              <span className="text-sm flex items-center gap-1">
                <Share2 className="h-4 w-4" /> Share:
              </span>
              <button 
                onClick={() => handleShare('linkedin')}
                className="p-2 rounded-full bg-[#0077B5]/10 hover:bg-[#0077B5]/20 text-[#0077B5] transition-colors"
              >
                <Linkedin className="h-4 w-4" />
              </button>
              <button 
                onClick={() => handleShare('twitter')}
                className="p-2 rounded-full bg-[#1DA1F2]/10 hover:bg-[#1DA1F2]/20 text-[#1DA1F2] transition-colors"
              >
                <Twitter className="h-4 w-4" />
              </button>
              <button 
                onClick={() => handleShare('facebook')}
                className="p-2 rounded-full bg-[#4267B2]/10 hover:bg-[#4267B2]/20 text-[#4267B2] transition-colors"
              >
                <Facebook className="h-4 w-4" />
              </button>
            </div>
          </div>
        </header>

        {/* Cover Image */}
        {blog.cover_image && (
          <div className="relative h-64 md:h-96 lg:h-[500px] rounded-2xl overflow-hidden mb-10 shadow-xl">
            <Image
              src={blog.cover_image}
              alt={blog.title}
              fill
              className="object-cover"
              priority
            />
          </div>
        )}

        {/* Article Content */}
        <div 
          className="prose prose-lg max-w-none prose-headings:text-foreground prose-p:text-muted-foreground prose-strong:text-foreground prose-a:text-primary"
          dangerouslySetInnerHTML={{ __html: formatContent(blog.content) }}
        />

        {/* Author Box */}
        <div className="mt-12 p-6 bg-card rounded-2xl border">
          <div className="flex items-start gap-4">
            {blog.author_avatar ? (
              <Image
                src={blog.author_avatar}
                alt={blog.author_name}
                width={64}
                height={64}
                className="rounded-full"
              />
            ) : (
              <div className="w-16 h-16 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0">
                <User className="h-8 w-8 text-primary" />
              </div>
            )}
            <div>
              <h3 className="font-semibold text-lg text-foreground">{blog.author_name}</h3>
              <p className="text-muted-foreground text-sm">Healthcare Professional at Ahericare</p>
            </div>
          </div>
        </div>
      </article>

      {/* Related Articles */}
      {relatedBlogs.length > 0 && (
        <section className="container mx-auto px-4 py-16 max-w-6xl">
          <h2 className="text-2xl font-bold text-foreground mb-8">Related Articles</h2>
          <div className="grid md:grid-cols-3 gap-6">
            {relatedBlogs.map((related) => (
              <Link 
                key={related.id}
                href={`/blog/${related.slug}`}
                className="group bg-card rounded-xl overflow-hidden shadow hover:shadow-lg transition-all"
              >
                <div className="relative h-40">
                  {related.cover_image ? (
                    <Image
                      src={related.cover_image}
                      alt={related.title}
                      fill
                      className="object-cover group-hover:scale-105 transition-transform duration-300"
                    />
                  ) : (
                    <div className="w-full h-full bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
                      <span className="text-3xl text-primary/40">📝</span>
                    </div>
                  )}
                </div>
                <div className="p-4">
                  <h3 className="font-semibold text-foreground group-hover:text-primary transition-colors line-clamp-2">
                    {related.title}
                  </h3>
                  {related.published_at && (
                    <p className="text-sm text-muted-foreground mt-2">
                      {format(new Date(related.published_at), 'MMM d, yyyy')}
                    </p>
                  )}
                </div>
              </Link>
            ))}
          </div>
        </section>
      )}
    </main>
  );
}

function formatContent(content: string): string {
  // Convert markdown-like content to HTML
  return content
    .split('\n\n')
    .map(paragraph => {
      // Handle headers
      if (paragraph.startsWith('### ')) {
        return `<h3>${paragraph.slice(4)}</h3>`;
      }
      if (paragraph.startsWith('## ')) {
        return `<h2>${paragraph.slice(3)}</h2>`;
      }
      if (paragraph.startsWith('# ')) {
        return `<h1>${paragraph.slice(2)}</h1>`;
      }
      // Handle lists
      if (paragraph.includes('\n- ')) {
        const items = paragraph.split('\n- ').filter(Boolean);
        return `<ul>${items.map(item => `<li>${item}</li>`).join('')}</ul>`;
      }
      // Regular paragraph
      return `<p>${paragraph}</p>`;
    })
    .join('');
}
